/*
 * cIPEndpoint.cpp
 *
 * An IP endpoint.
 */
#include "cSimEndpoint.h"
#include "Endpoint/cEndpointFactory.h"
#include "Util/CoolHash.h"
#include <memory.h>

cSimEndpoint::cSimEndpoint()
{
	mHashCode = 0;
	mType = ENDPOINT_TYPE_SIM;
	mRefCount = 0;
	mSubnetIndex   = 0;
	mAddressIndex  = 0;
}

cSimEndpoint::cSimEndpoint(unsigned int subnet, unsigned int address)
{
	mHashCode = 0;
	mType = ENDPOINT_TYPE_SIM;
	mRefCount = 0;
	mSubnetIndex  = subnet;
	mAddressIndex = address;
}

cSimEndpoint::~cSimEndpoint()
{

}

unsigned int cSimEndpoint::HashCode()
{
	if(!mHashCode)	// re-compute if hashcode is 0
	{
		_ComputeHashCode();
	}
	return mHashCode;
}

unsigned int cSimEndpoint::_ComputeHashCode()
{
	unsigned char*  ptr;
	unsigned char	x[sizeof(mSubnetIndex)+sizeof(mAddressIndex)];
	ptr = &x[0];
	memcpy(ptr, &mSubnetIndex, sizeof(mSubnetIndex));
	ptr += sizeof(mSubnetIndex);
	memcpy(ptr, &mAddressIndex, sizeof(mAddressIndex));
	mHashCode = CoolHash(&x[0], sizeof(x));
	return mHashCode;
}

/*
 * cSimEndpoint::Print()
 *
 * Purpose:	Prints out the endpoint to the given stream (for debug purposes)
 * IN:		errStream	-> The stream to use for printing.
 * OUT:		-
 * Cond:	-
 * PostCnd:	-
 * Return:	-
 */
void cSimEndpoint::Print(ostream &errStream)
{
	errStream << "@@@@@@@@@@@@@@@@@@@" << endl;
	errStream << "Subnet:   " << mSubnetIndex  << endl;
	errStream << "Address:  " << mAddressIndex << endl;
	errStream << "Hash: " << HashCode() << endl;
	errStream << "This: " << this << endl;
	errStream << "RefC: " << mRefCount << endl;
	errStream << "@@@@@@@@@@@@@@@@@@@" << endl;
}


/*
 * cSimEndpoint::Serialize()
 *
 * Purpose:	Turns the endpoint into a byte stream from in the given buffer.
 * IN:		buffer		-> The buffer in which to place serialized endpoint.
 *			size		-> The size of the buffer provided.
 * OUT:		size		-> The size remaining in the buffer, or the size needed if fail
 * Cond:	-
 * PostCnd:	The endpoint is in the buffer.
 * Return:	The address of the location in the buffer after the serialized representation
 *			OR NULL if failure.
 */
char* cSimEndpoint::Serialize(char* buffer, int *size)
{
 	int sizeNeeded = GetSize();
	if(*size < sizeNeeded)
	{
		*size = sizeNeeded;
		return NULL;
	}
	memcpy(buffer, &mType, sizeof(mType));
	buffer += sizeof(mType);
	memcpy(buffer, &mSubnetIndex, sizeof(mSubnetIndex));
	buffer += sizeof(mSubnetIndex);
	memcpy(buffer, &mAddressIndex, sizeof(mAddressIndex)); 
	buffer += sizeof(mAddressIndex);
	*size = *size - sizeNeeded;
	return buffer;
}

/*
 * cSimEndpoint::Deserialize()
 *
 * Purpose:	Fills in the endpoint information from the given buffer.
 * IN:		buffer		-> The buffer in which the endpoint info resides.
 *			size		-> The size of the buffer provided.
 * OUT:		size		-> The size remaining in the buffer, or the size needed if fail
 * Cond:	A valid endpoint of this type is in the buffer.
 * PostCnd:	This endpoint is set up.
 * Return:	The address of the location in the buffer after the serialized representation
 *			OR NULL if failure.
 */
char* cSimEndpoint::Deserialize(char *buffer, int *size)
{
	int sizeNeeded = GetSize();
	if(*size < sizeNeeded)
	{
		*size = sizeNeeded;
		return NULL;
	}
	memcpy(&mType, buffer, sizeof(mType));
	buffer += sizeof(mType);
	memcpy(&mSubnetIndex, buffer, sizeof(mSubnetIndex));
	buffer += sizeof(mSubnetIndex);
	memcpy(&mAddressIndex, buffer, sizeof(mAddressIndex));
	buffer += sizeof(mAddressIndex);
	*size = *size - sizeNeeded;
	_ComputeHashCode();

	return buffer;
}

unsigned int cSimEndpoint::GetSize()
{
	return sizeof(mAddressIndex) + sizeof(mSubnetIndex) + sizeof(mType);
}

bool cSimEndpoint::IsSameSubnet(cEndpoint* ep) 
{
	cSimEndpoint* iep;
	if(ep->GetType() == ENDPOINT_TYPE_SIM)
	{
		iep = (cSimEndpoint *)ep;
		return (iep->mSubnetIndex == mSubnetIndex);
	}
	return false;
}

bool cSimEndpoint::Equals(cHashObject* hObj)
{
	cSimEndpoint* ep = (cSimEndpoint *)hObj;
	if(ep->GetType() != ENDPOINT_TYPE_SIM)
	{
		return false;
	}
	else if(ep->HashCode() != HashCode())
	{
		return false;
	}
	else
	{
		return ( (mSubnetIndex == ep->mSubnetIndex) && (mAddressIndex == ep->mAddressIndex) );
	}
}

bool cSimEndpoint::SubnetLessThan(cEndpoint* ep)
{
	if(ep->GetType() != ENDPOINT_TYPE_SIM)
	{
		return false;
	}

	cSimEndpoint* iep = (cSimEndpoint *)ep;
	return (mSubnetIndex < iep->mSubnetIndex);
}

bool cSimEndpoint::LessThan(cHashObject* hObj)
{
	cSimEndpoint* ep = (cSimEndpoint *)hObj;
	if(HashCode() < ep->HashCode())
	{
		return true;
	}
	else if(HashCode() > ep->HashCode())
	{
		return false;
	}
	else
	{
		if(mSubnetIndex < ep->mSubnetIndex)
		{
			return true;
		}
		else if(mSubnetIndex > ep->mSubnetIndex)
		{
			return false;
		}
		else
		{
			return (mAddressIndex < ep->mAddressIndex);
		}
	}
}

cEndpoint*	cSimEndpoint::AllocCopy()
{
	cSimEndpoint* retVal;
	retVal = (cSimEndpoint *)cEndpointFactory::AllocEndpoint(mType);
	if(retVal)
	{
		*retVal = *this;	// Shallow copy
	}
	return retVal;
}
